# MOP4Import::Base::CLI_JSON の存在意義

## 概要

MOP4Import::Base::CLI_JSON（以下 CLI_JSON）は、Perl モジュールの開発を加速するための基底クラスです。このモジュールの最大の価値は、**開発中のモジュールの任意のメソッドをコマンドラインから即座にテスト可能にする**ことにあります。

## 解決する問題

### 従来のモジュール開発の課題

Perl モジュールを開発する際、通常は以下のような手順を踏みます：

1. モジュールを書く
2. テストスクリプトを書く
3. テストを実行して動作確認
4. 問題があれば修正して再テスト

この開発サイクルでは、メソッドを一つ書くたびにテストスクリプトを書かないと動作確認ができません。特に開発初期の探索的な段階では、このオーバーヘッドが開発速度を大きく低下させます。

### CLI_JSON が提供する解決策

CLI_JSON を継承したモジュールでは、以下のような開発フローが可能になります：

```perl
#!/usr/bin/env perl
package MyScript;
use MOP4Import::Base::CLI_JSON -as_base;
MY->cli_run(\@ARGV) unless caller;

sub query {
  my ($self, $sql) = @_;
  # SQLクエリの実装
}
1;
```

このモジュールは即座にコマンドラインから実行できます：

```bash
# 構文チェック
$ perl -wc MyScript.pm

# すぐに実行
$ ./MyScript.pm query "SELECT * FROM users"

# デバッガで実行
$ perl -d ./MyScript.pm query "SELECT * FROM users"

# プロファイラで実行
$ perl -d:NYTProf ./MyScript.pm query "SELECT * FROM users"
```

## 設計の核心

### 1. JSON による統一的なデータ交換

引数と戻り値を JSON でシリアライズ可能なものに制限することで、以下が実現します：

- **複雑なデータ構造の受け渡し**: 配列やハッシュを自然に扱える
- **言語中立なインターフェース**: 他の言語やツールとの連携が容易
- **デバッグの容易さ**: 入出力が人間に読みやすい形式

### 2. 出力形式の柔軟性

デフォルトの NDJSON（Newline Delimited JSON）を含む複数の出力形式をサポート：

- **ndjson**: 巨大な配列でも読みやすく、grep などの行指向ツールと連携しやすい
- **json**: 標準的な JSON 形式
- **yaml**: より人間に読みやすい形式
- **tsv**: 表形式データの処理に便利
- **dump**: Perl のData::Dumperによるデバッグ出力

### 3. OO Modulino パターンの実現

同一ファイルが以下の３つの役割を果たせます：

1. **通常のPerlモジュール**: `use MyScript;` として利用
2. **CLIツール**: `./MyScript.pm` として実行
3. **テスト対象**: 任意のメソッドを個別にテスト

## 開発効率の向上

### 探索的開発の支援

メソッドを書いたら即座にコマンドラインから試せるため：

- テストスクリプトを書く前に動作を確認できる
- 試行錯誤が高速に行える
- インターフェースの設計を実際に使いながら洗練できる

### テスタビリティの向上

コマンドラインから呼び出しやすい単位でメソッドを切り分けることが自然に促され、結果として：

- 単体テストが書きやすいメソッド設計になる
- 責務が明確に分離される
- 再利用しやすいコードになる

### デバッグとプロファイリングの容易さ

標準的な Perl のツールがそのまま使えます：

- `perl -d` でデバッガ起動
- `perl -d:NYTProf` でプロファイリング
- `perl -MDevel::Trace` でトレース

## まとめ

CLI_JSON の存在意義は、**モジュール開発のフィードバックループを劇的に短縮する**ことにあります。これは単なる CLI ツール作成フレームワークではなく、Perl モジュール開発の生産性を向上させるための開発支援ツールなのです。

## 関連文書

- [OO Modulino パターンについて](./OO_Modulino.md)
- [CLI_JSON リファレンスマニュアル](../Base/CLI_JSON.pod)